﻿
/****************************************************************************/
/*Copyright (c) 2011, Florent DEVILLE.                                      */
/*All rights reserved.                                                      */
/*                                                                          */
/*Redistribution and use in source and binary forms, with or without        */
/*modification, are permitted provided that the following conditions        */
/*are met:                                                                  */
/*                                                                          */
/* - Redistributions of source code must retain the above copyright         */
/*notice, this list of conditions and the following disclaimer.             */
/* - Redistributions in binary form must reproduce the above                */
/*copyright notice, this list of conditions and the following               */
/*disclaimer in the documentation and/or other materials provided           */
/*with the distribution.                                                    */
/* - The names of its contributors cannot be used to endorse or promote     */
/*products derived from this software without specific prior written        */
/*permission.                                                               */
/* - The source code cannot be used for commercial purposes without         */
/*its contributors' permission.                                             */
/*                                                                          */
/*THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS       */
/*"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT         */
/*LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS         */
/*FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE            */
/*COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,       */
/*INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,      */
/*BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;          */
/*LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER          */
/*CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT        */
/*LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN         */
/*ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE           */
/*POSSIBILITY OF SUCH DAMAGE.                                               */
/****************************************************************************/

using System;
using Microsoft.Xna.Framework;
using GE.World.Entities;

namespace GE.Physics.Shapes
{
    /// <summary>
    /// Rectangle shape used for static entity
    /// </summary>
    class StaticShapeRectangle : ShapeRectangle
    {
        /// <summary>
        /// Store the four points of the rectangle with oriented axis
        /// </summary>
        private Vector2[] _v2Obb;

        /// <summary>
        /// Constructor. Set the Id to -1.
        /// </summary>
        /// <param name="width">Rectangle's width</param>
        /// <param name="height">Rectangle's height</param>
        /// <param name="rotationCenter">Point used as rotation center (origin used for the shape)</param>
        /// <param name="position">Position in the world</param>
        /// <param name="orientation">Orientation in the world</param>
        /// <param name="owner">Entity owner of the shape</param>
        public StaticShapeRectangle(int width, int height, Vector2 rotationCenter, Vector2 position, float orientation, WorldEntity owner)
            :base(width, height, rotationCenter, position, orientation, owner)
        {
            _iId = -1;
            _v2Obb = new Vector2[4];
            _bCollisionEnable = true;

            computeBoundingBox();
        }

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="id">Id of the shape. Must be uniqe and should be generated by the Physics component</param>
        /// <param name="width">Rectangle's width</param>
        /// <param name="height">Rectangle's height</param>
        /// <param name="rotationCenter">Point used as rotation center (origin used for the shape)</param>
        /// <param name="position">Position in the world</param>
        /// <param name="orientation">Orientation in the world</param>
        /// <param name="owner">Entity owner of the shape</param>
        public StaticShapeRectangle(int id, int width, int height, Vector2 rotationCenter, Vector2 position, float orientation, WorldEntity owner)
            : this(width, height, rotationCenter, position, orientation, owner)
        {
            _iId = id;
            //_v2Obb = new Vector2[4];
            //_bCollisionEnable = true;

            //computeBoundingBox();
        }

        /// <summary>
        /// Compute the oriented axis bounding box
        /// </summary>
        private void computeBoundingBox()
        {

            //compute the aligned axis bounding box
            Vector2[] aabb = new Vector2[4];
            aabb[0] = new Vector2((float)-_v2RotationCenter.X, -(float)_v2RotationCenter.Y);
            aabb[1] = new Vector2((float)-_v2RotationCenter.X + _iWidth, -(float)_v2RotationCenter.Y);
            aabb[2] = new Vector2((float)-_v2RotationCenter.X + _iWidth, -(float)_v2RotationCenter.Y + _iHeight);
            aabb[3] = new Vector2((float)-_v2RotationCenter.X, -(float)_v2RotationCenter.Y + _iHeight);

            //compute the oriented axis bounding box
            for (int i = 0; i < 4; i++)
            {
                float cosinus = (float)Math.Cos(_fOrientation);
                float sinus = (float)Math.Sin(_fOrientation);

                _v2Obb[i].X = cosinus * aabb[i].X - sinus * aabb[i].Y + _v2position.X;

                _v2Obb[i].Y = cosinus * aabb[i].Y + sinus * aabb[i].X + _v2position.Y;

            }
        }

        /// <summary>
        /// Return the oriented bounding box
        /// </summary>
        /// <returns></returns>
        public override Vector2[] getOrientedBoundingBox()
        {
            return _v2Obb;
        }


    }
}
